;/////////////////////////////////////////////////////////////////////////////////////////////////
;// Arduino Mega (atMega1280) Logic Analyzer (AMLA) Project
;// SoaSystem Engineering - 2 November 2011
;//
;/////////////////////////////////////////////////////////////////////////////////////////////////
;// Version 1.0.0 - currently only support:
;// - rise/fall single trigger mode on all PINK
;// - 2MHz sampling rate on 8 channel, ~8KB/ch RAM
;// - 1MHz sampling rate on 16 channel ~4KB/ch RAM
;//
;/////////////////////////////////////////////////////////////////////////////////////////////////
;// procedure:
;// 1) on AMLA startup, it will listen PC transmitting data valued 170
;// 2) AMLA will send header/synch/id data when there's PC data 170 received.
;// 3) AMLA will enter single shot triggering mode
;// 4) when triggered, AMLA will send 124 packets containing 64 bytes in each packet
;//    packet format = 64 bytes LA data in SRAM
;//    124 packets containing 62 x 124 = 7936 bytes of memory depth
;// 5) at each packet send completion, AMLA will wait for any conformation signal (not 170) from
;//    PC before continuing to send the next packet, this ensure usb/uart comm stability.
;//    after sending the last packet, no listening/sending is required.
;// 6) AMLA will repeat step 1 - 5 until resetted
;//
;/////////////////////////////////////////////////////////////////////////////////////////////////

#define _ATMEGA1280_UART_ENABLE
.include "avr_atmega1280_asm.inc"

;temporary usage
.def tmp = r21
.def dmp = r22

;defined to use 8 bit 2MHz sampling
;undefine to use 16 bit 1MHz sampling
#define FASTEST_SAMPLING



;#define DEBUGGING
;for debugging (PINB)
#ifdef DEBUGGING
.def ledOn = r23
.def ledOf = r24
#endif



;/////////////////////////////////////////////////////////////////////////////////////////////////
;// macro

.macro sendMyUartL;(val)
    sendUartL 0, @0, 0, 1
.endm

.macro sendMyUartF;(reg)
    sendUartF 0, @0, 0, 1
.endm



;/////////////////////////////////////////////////////////////////////////////////////////////////
;// setup (16 pin LA input)

init:

    ;enable arduino-usb uart0
    enableUartL 0

    ;default mega1280 8 pin     - PINK 0 - 7
    ;arduino analog pin mapping - ANA  8 - 15
    ;AMLA pin mapping           - INP  7 - 0

    setAsInput H, K, 0
    setAsInput H, K, 1
    setAsInput H, K, 2
    setAsInput H, K, 3
    setAsInput H, K, 4
    setAsInput H, K, 5
    setAsInput H, K, 6
    setAsInput H, K, 7

    ;mega1280 extended 16 pin   - PINF 0 - 7
    ;arduino analog pin mapping - ANA  0 - 7
    ;AMLA pin mapping           - INP 15 - 8

    setAsInput R, F, 0
    setAsInput R, F, 1
    setAsInput R, F, 2
    setAsInput R, F, 3
    setAsInput R, F, 4
    setAsInput R, F, 5
    setAsInput R, F, 6
    setAsInput R, F, 7

    ;enable all interrupt mask for PINK to enable PCINT (PCIFR, PCIF2) flagging
    setValueHL PCMSK2, 0xFF

    ;for debugging (PINB)
    #ifdef DEBUGGING
    call init_debug
    #endif
return


;/////////////////////////////////////////////////////////////////////////////////////////////////
;// debug program

#ifdef DEBUGGING

;set all portb as output
init_debug:
    setAsOutput R, B, 0
    setAsOutput R, B, 1
    setAsOutput R, B, 2
    setAsOutput R, B, 3
    setAsOutput R, B, 4
    setAsOutput R, B, 5
    setAsOutput R, B, 6
    setAsOutput R, B, 7
    setValueFL ledOn, 0b11111111
    setValueFL ledOf, 0b00000000
return

;wait incoming from pc -> usb -> uart
wait_uart_debug:
    call wait_uart
    setValueRF PORTB, ledOn
    setValueRF PORTB, ledOf
    goto wait_uart_debug
return

#endif // DEBUGGING



;/////////////////////////////////////////////////////////////////////////////////////////////////
;// main application entry

main:
    call init

    #ifdef DEBUGGING
    call wait_uart_debug
    #endif

    loop:

        ;restart synchronization (to enable single trigger mode)
        call synchro
        flushUartL 0

        ;wait for a new trigger (PINK change interrupt)
        setBitR PCIFR, PCIF2
        waitIntr:
            skipIfBitSetR PCIFR, PCIF2
        goto waitIntr
        call start_memory

        ;8ch 2MSps sampler
        #ifdef FASTEST_SAMPLING
        call start_sampling_8
        #else

        ;16ch 1MSps sampler
        call start_sampling_16
        #endif

		;send to usb
        call start_memory
		call send_sample
    goto loop
ends:



;/////////////////////////////////////////////////////////////////////////////////////////////////
;// util

send_id:
    sendMyUartL 83
    sendMyUartL 79
    sendMyUartL 65
    sendMyUartL 83
    sendMyUartL 89
    sendMyUartL 83
    sendMyUartL 84
    sendMyUartL 69
    sendMyUartL 77
return

;send unique header for synchronization data to uart
synchro:
    call wait_uart
    skipIfEquWL 170
    goto synchro
    call send_id
return

;wait incoming from pc -> usb -> uart
wait_uart:
    waitUartReceivedL 0
    readUartL 0
return

;set pointer to the beginning of SRAM memory
start_memory:
    setValueFL YH, HIGH(SRAM_START)
    setValueFL YL, LOW(SRAM_START)
return



;/////////////////////////////////////////////////////////////////////////////////////////////////
;// 8 pin sampler - 16MHz / 8 clock = 2MSps MAX

.macro nop5
    nop
    nop
    nop
    nop
    nop
.endm

.macro nop25
    nop5
    nop5
    nop5
    nop5
    nop5
.endm

.macro nop50
    nop25
    nop25
.endm

.macro nop72
    nop50
    nop5
    nop5
    nop5
    nop5
    nop
    nop
.endm

;to reduce sampling rate to (16 / (8 + N)) MSps
;where N is this macro's opcodes count
;insert nops here

.macro reduce_sampling_8
    // uncomment below to make 200KSps (16MHz / 8 + 72 opcodes)
    // to be able to capture longer data (<= 20Kbps)
    // nop72
.endm

start_sampling_8:

    ;sample 8 channel PINK and store in SRAM
    lds W, PINK
    st Y+, W

    ;to reduce sampling rate if required
    reduce_sampling_8

    ;quit if 7936 SRAM are written (must leave some SRAM for stack)
    skipIfGrEqFL YH, 33
    goto start_sampling_8
return



;/////////////////////////////////////////////////////////////////////////////////////////////////
;// 16 pin sampler - 16MHz / 11 clock = 1.455MSps MAX

;to reduce sampling rate to (16 / (11 + N)) MSps
;where N is this macro's opcodes count
;insert nops here

;5 nop to round off to a nice 1MSps
.macro reduce_sampling_16
    nop
    nop
    nop
    nop
    nop
.endm

start_sampling_16:

    ;sample 8 channel PINK and store in SRAM
    lds W, PINK
    st Y+, W

    ;sample 8 channel PINF and store in SRAM
    in W, PINF
    st Y+, W

    ;to reduce sampling rate if required
    reduce_sampling_16

    ;quit if 7936 SRAM are written (must leave some SRAM for stack)
    skipIfGrEqFL YH, 33
    goto start_sampling_16
return



;to be used by send_sample 
#define MEMSIZE 7936
#define PCKSIZE 64
#define PCKCNTR INT(MEMSIZE / PCKSIZE)

;/////////////////////////////////////////////////////////////////////////////////////////////////
;// send 7936 bytes SRAM/LA data to uart -> usb -> pc app in 64 bytes packet (124 packets)

send_sample:

    ;send la packets
    zeroValueF dmp

    ;send data incrementally from SRAM
    start_send_sample:
    forFL tmp, 0
    send_sample_packet:
        ld var2, Y+
        sendMyUartF var2
    nextFLL tmp, PCKSIZE, 1, send_sample_packet

    ;check if this is the end of packet
    incValueF dmp
    skipIfLessFL dmp, PCKCNTR
    return

    ;still got packet. listen before resume sending
    call wait_uart
goto start_send_sample
